package com.tchudyk.text_canvas;

import com.tchudyk.text_canvas.layout.LayoutEngine;
import com.tchudyk.text_canvas.layout.LayoutLine;
import com.tchudyk.text_canvas.renderer.FxRenderer;
import com.tchudyk.text_canvas.segment.Segment;
import javafx.beans.property.ObjectProperty;
import javafx.beans.property.SimpleObjectProperty;
import javafx.geometry.BoundingBox;
import javafx.geometry.Bounds;
import javafx.geometry.Insets;
import javafx.scene.canvas.Canvas;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.input.KeyEvent;
import javafx.scene.input.MouseEvent;
import javafx.scene.input.ScrollEvent;
import javafx.scene.layout.Region;
import javafx.scene.paint.Color;

import java.util.ArrayList;
import java.util.List;

public class TextCanvas extends Region {
    private static final double PADDING_LEFT = 20;
    private static final double PADDING_RIGHT = 20;
    private static final double PADDING_TOP = 20;

    private final List<Segment> segments = new ArrayList<>();
    private final FxNodes nodesRegistry;
    private final NodeMeasurer measurer = new FxNodeMeasurer();
    private final ObjectProperty<Bounds> visibleBounds = new SimpleObjectProperty<>();

    private final Canvas canvas;
    private final List<LayoutLine> renderLines = new ArrayList<>();

    public TextCanvas() {

        canvas = new Canvas(800, 400);
        nodesRegistry = new FxNodes(getChildren(), this::repaint);
        setFocusTraversable(true);
        getChildren().add(canvas);

        // simple resize handling
        canvas.widthProperty().bind(widthProperty());
        canvas.heightProperty().bind(heightProperty());
        canvas.widthProperty().addListener(o -> moveBounds(0, 0));
        canvas.heightProperty().addListener(o -> moveBounds(0, 0));
        visibleBounds.addListener(observable -> repaint());

        visibleBounds.setValue(new BoundingBox(0, 0, getWidth(), getHeight()));

        addEventFilter(KeyEvent.KEY_PRESSED, event -> {
            switch (event.getCode()) {
                case LEFT -> moveBounds(-10, 0);
                case RIGHT -> moveBounds(10, 0);
                case UP -> moveBounds(0, -10);
                case DOWN -> moveBounds(0, 10);
            }
        });
        addEventHandler(MouseEvent.MOUSE_PRESSED, event -> {
            requestFocus();
        });
        addEventHandler(ScrollEvent.SCROLL, event -> {
            double deltaX = event.getDeltaX();
            double deltaY = event.getDeltaY() * -1;
            moveBounds(deltaX, deltaY);
            event.consume();
        });
    }

    private void moveBounds(double deltaX, double deltaY) {
        Bounds bounds = visibleBounds.get();
        visibleBounds.set(
                new BoundingBox(bounds.getMinX() + deltaX, bounds.getMinY() + deltaY, getWidth(), getHeight())
        );
    }

    public void addSegment(Segment segment) {
        segments.add(segment);
    }

    public void repaint() {
        calculateSegmentsTokens();
        redrawFrame();
    }

    private void redrawFrame() {
        // Clear area
        GraphicsContext gc = canvas.getGraphicsContext2D();
        gc.setFill(Color.WHITE);
        gc.fillRect(0, 0, canvas.getWidth(), canvas.getHeight());

        Bounds visibleBounds = this.visibleBounds.get();
        FxRenderer fxRenderer = new FxRenderer(gc, nodesRegistry);
        fxRenderer.render(renderLines, visibleBounds);
    }

    private void calculateSegmentsTokens() {
        LayoutEngine layoutEngine = new LayoutEngine(PADDING_LEFT, PADDING_TOP, canvas.getWidth() - (PADDING_LEFT + PADDING_RIGHT), measurer);
        for (Segment seg : segments) {
            layoutEngine.appendSegment(seg);
        }
        renderLines.clear();
        renderLines.addAll(layoutEngine.getLines());
    }

    @Override
    protected void layoutChildren() {
        super.layoutChildren();
        Insets insets = getInsets();
        double width = getWidth();
        double height = getHeight();
        canvas.resizeRelocate(
                insets.getLeft(),
                insets.getTop(),
                width - insets.getLeft() - insets.getRight(),
                height - insets.getTop() - insets.getBottom()
        );
    }

    @Override
    protected double computePrefWidth(double height) {
        return canvas.prefWidth(height);
    }

    @Override
    protected double computePrefHeight(double width) {
        return canvas.prefHeight(width);
    }

    public ObjectProperty<Bounds> visibleBoundsProperty() {
        return visibleBounds;
    }
}
