package com.tchudyk.text_canvas.layout;

import com.tchudyk.text_canvas.NodeMeasurer;
import com.tchudyk.text_canvas.segment.Segment;
import com.tchudyk.text_canvas.segment.SegmentStyle;
import com.tchudyk.text_canvas.segment.blocks.BlockSegment;
import com.tchudyk.text_canvas.segment.blocks.InlineBlockSegment;
import com.tchudyk.text_canvas.segment.node.NodeSegment;
import com.tchudyk.text_canvas.segment.text.TextSegment;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.regex.Pattern;

public class LayoutEngine {

    private static final Pattern SPLIT_PATTERN = Pattern.compile("(?<=\\s)");

    private final NodeMeasurer nodeMeasurer;
    private final Deque<LayoutLine> lines = new ArrayDeque<>();
    private LayoutLine currentLine;
    private int wordStartSegment;
    private double beforeWordStartAscent;
    private double beforeWordStartDescent;
    private boolean isEndOfWord = true;

    private double lastX;
    private double lastY;
    private double lineHeightAscent;
    private double lineHeightDescent;
    public final double maxWidth; // Waring: This is maxWidth, not maxX. (MaxX = startX + maxWidth)
    public double maxDiscoveredX;
    public final double startX;

    public LayoutEngine(double startX, double startY, double maxWidth, NodeMeasurer nodeMeasurer) {
        this.startX = startX;
        this.lastX = startX;
        this.lastY = startY;
        this.lineHeightAscent = 0;
        this.lineHeightDescent = 0;
        this.maxWidth = maxWidth;
        this.currentLine = new LayoutLine(lastY, lineHeightAscent, lineHeightDescent);
        this.nodeMeasurer = nodeMeasurer;
    }

    public void appendSegment(Segment segment) {
        if (segment instanceof TextSegment textSegment) {
            handleTextSegment(textSegment);
        } else if (segment instanceof NodeSegment nodeSegment) {
            handleNodeSegment(nodeSegment);
        } else if (segment instanceof InlineBlockSegment blockSegment) {
            handleInlineBlockSegment(blockSegment);
        } else if (segment instanceof BlockSegment blockSegment) {
            handleBlockSegment(blockSegment);
        }
    }

    private void handleInlineBlockSegment(InlineBlockSegment segment) {
        // PHASE 1: First calculation - to discover the max width of the block
        LayoutEngine innerEngine = new LayoutEngine(0, lastY, maxWidth, nodeMeasurer);
        for (Segment seg : segment.childSegments()) {
            innerEngine.appendSegment(seg);
        }

        // PHASE 2: Calculate final block width
        double availableWidthInCurrentLine = maxWidth - lastX;
        double discoveredWidth = innerEngine.maxDiscoveredX;

        if (discoveredWidth > availableWidthInCurrentLine) {
            commitLine(lineHeightAscent, lineHeightDescent);
        }

        double correctBlockWidth = Math.min(discoveredWidth, maxWidth - lastX);

        // PHASE 3: Second calculation - to fit to the given width
        innerEngine = new LayoutEngine(lastX, lastY, correctBlockWidth, nodeMeasurer);
        for (Segment seg : segment.childSegments()) {
            innerEngine.appendSegment(seg);
        }

        // PHASE 4: Update layout with inner layout
        double height = (innerEngine.lastY - lastY) + innerEngine.lineHeightAscent; // ascentHeight - to make text match baseLine (like in a web browser)
        double width = innerEngine.maxDiscoveredX - lastX;

        SegmentReference currentToken = new SegmentReference(segment, (short) 0, (short) 1, lastX, width, height);
        currentLine.segments.add(currentToken);

        jumpToX(innerEngine.maxDiscoveredX);
        lineHeightAscent = Math.max(height, lineHeightAscent);
        lineHeightDescent = Math.max(innerEngine.lineHeightDescent, lineHeightDescent);
        currentToken.updateInnerLines(new ArrayList<>(innerEngine.getLines()));

        isEndOfWord = true;
        wordStartSegment = currentLine.segments.size();
    }

    private void handleBlockSegment(BlockSegment blockSegment) {
        commitLine(lineHeightAscent, lineHeightDescent);

        LayoutEngine innerEngine = new LayoutEngine(lastX, lastY, maxWidth, nodeMeasurer);
        for (Segment seg : blockSegment.childSegments()) {
            innerEngine.appendSegment(seg);
        }
        double height = (innerEngine.lastY - lastY) + innerEngine.lineHeightAscent; // ascentHeight - to make text match baseLine (like in a web browser)
        double blockStartX = lastX;

        increaseX(innerEngine.maxDiscoveredX - lastX); // to register inner width
        jumpToX(innerEngine.lastX + maxWidth); // to jump to the end of the line
        lineHeightAscent = height;
        lineHeightDescent = innerEngine.lineHeightDescent;

        SegmentReference currentToken = new SegmentReference(blockSegment, (short) 0, (short) 1, blockStartX, maxWidth, height);
        currentToken.updateInnerLines(new ArrayList<>(innerEngine.getLines()));
        currentLine.segments.add(currentToken);

        isEndOfWord = true;
        wordStartSegment = currentLine.segments.size();
    }

    private void handleNodeSegment(NodeSegment segment) {

        double[] measured = nodeMeasurer.measure(segment.node);
        double nodeWidth = measured[0];
        double nodeHeight = measured[1];

        boolean shouldWrapLine = lastX + nodeWidth > startX + maxWidth;
        if (shouldWrapLine) {
            commitLine(lineHeightAscent, lineHeightDescent);
        }

        SegmentReference currentToken = new SegmentReference(segment, (short) 0, (short) 1, lastX, nodeWidth, nodeHeight);
        currentLine.segments.add(currentToken);

        increaseX(nodeWidth);
        lineHeightAscent = Math.max(lineHeightAscent, nodeHeight);
        lineHeightDescent = Math.max(lineHeightDescent, 0);

        isEndOfWord = true;
        wordStartSegment = currentLine.segments.size();
    }

    private void handleTextSegment(TextSegment segment) {
        String[] words = SPLIT_PATTERN.split(segment.text());
        SegmentReference currentToken = null;
        int textOffset = 0;

        for (int i = 0; i < words.length; i++) {
            String w = words[i];
            String restOfWord = null;
            boolean endsWithNewLine = w.endsWith("\n") || w.endsWith("\r");
            String displayText;
            if (endsWithNewLine) {
                displayText = w.substring(0, w.length() - 1);
            } else {
                displayText = w;
            }
            NodeMeasurer.TextMetrics textMetrics = measure(displayText, segment.style());
            double wordWidth = textMetrics.width();
            double ascent = textMetrics.ascentHeight();
            double descent = textMetrics.descentHeight();

            boolean shouldBeMovedToNextLine = lastX + wordWidth > startX + maxWidth;
            if (shouldBeMovedToNextLine && isEndOfWord) {
                if (currentToken != null) {
                    currentLine.segments.add(currentToken);
                    currentToken = null;
                }
                commitLine(lineHeightAscent, lineHeightDescent);
            } else if (shouldBeMovedToNextLine && !isEndOfWord) {
                if (currentToken != null) {
                    currentLine.segments.add(currentToken);
                    currentToken = null;
                }

                // Only text segments can be continued, so segments in the buffer are text segments
                List<SegmentReference> nodesToMove = currentLine.segments.subList(wordStartSegment, currentLine.segments.size());
                if (nodesToMove.isEmpty()) {
                    // It means that this is the first word in the line, and it's to big do fit its width.
                } else {
                    SegmentReference segmentToSplit = nodesToMove.get(0);
                    TextSegment textSegmentToSplit = (TextSegment) segmentToSplit.segment();
                    String lastTokenText = textSegmentToSplit.text().substring(segmentToSplit.offset(), segmentToSplit.offset() + segmentToSplit.limit());
                    int splitPosition = lastWhitespaceIndex(lastTokenText);
                    ArrayList<SegmentReference> nodesForNextLine = new ArrayList<>();
                    if (splitPosition > 0) {
                        // Split the first segment in nodesToMove
                        splitPosition += 1; // +1 (split after whitespace)
                        NodeMeasurer.TextMetrics removedMetrics = measure(lastTokenText.substring(splitPosition), textSegmentToSplit.style());
                        segmentToSplit.updateLimit(segmentToSplit.limit() - (lastTokenText.length() - splitPosition), segmentToSplit.width() - removedMetrics.width());
                        SegmentReference addedSegment = new SegmentReference(segmentToSplit.segment(), (short) (segmentToSplit.offset() + splitPosition), (short) (lastTokenText.length() - splitPosition), startX, removedMetrics.width(), removedMetrics.ascentHeight());
                        nodesForNextLine.add(addedSegment);
                        List<SegmentReference> restOfNodesToMove = nodesToMove.subList(1, nodesToMove.size());
                        nodesForNextLine.addAll(restOfNodesToMove);
                        restOfNodesToMove.clear();
                    } else {
                        // Move all notedToMove to the next line
                        nodesForNextLine.addAll(nodesToMove);
                        nodesToMove.clear();
                    }

                    commitLine(beforeWordStartAscent, beforeWordStartDescent);
                    currentLine.segments.addAll(nodesForNextLine);

                    // Recalculate new line positions
                    for (SegmentReference segmentReference : nodesForNextLine) {
                        segmentReference.updateX(lastX);
                        TextSegment t = (TextSegment) segmentReference.segment();
                        NodeMeasurer.TextMetrics prevMetrics = measure(t.text().substring(segmentReference.offset(), segmentReference.offset() + segmentReference.limit()), t.style());
                        increaseX(prevMetrics.width());
                        ascent = Math.max(ascent, prevMetrics.ascentHeight());
                        descent = Math.max(descent, prevMetrics.descentHeight());
                    }
                }
            }

            boolean isBiggerThanLineAfterSplit = lastX + wordWidth > startX + maxWidth;
            if (isBiggerThanLineAfterSplit) {
                NodeMeasurer.TextMetrics wordPartMetrics;
                String wordPart = displayText;
                int wordIndex = displayText.length() - 1;
                for (; wordIndex >= 0; wordIndex--) {
                    wordPart = displayText.substring(0, wordIndex);
                    wordPartMetrics = measure(wordPart, segment.style());
                    double partWidth = wordPartMetrics.width();
                    boolean isFitting = lastX + partWidth < startX + maxWidth;
                    if (isFitting) {
                        break;
                    }
                }
                // If we can split the word into smaller parts, do this
                if (wordIndex > 0) {
                    restOfWord = displayText.substring(wordIndex);
                    w = wordPart;
                    displayText = wordPart;
                } else if (wordIndex == 0 && !currentLine.segments.isEmpty()) {
                    // Split the line before this word (when it's not empty) and try it again
                    commitLine(lineHeightAscent, lineHeightDescent);
                    i -= 1;
                    continue;
                }
            }

            lineHeightAscent = Math.max(lineHeightAscent, ascent);
            lineHeightDescent = Math.max(lineHeightDescent, descent);
            if (!w.isEmpty() && lastWhitespaceIndex(w) == w.length() - 1) {
                isEndOfWord = true;
                wordStartSegment = currentLine.segments.size() + 1;
                beforeWordStartAscent = lineHeightAscent;
                beforeWordStartDescent = lineHeightDescent;
            } else {
                isEndOfWord = false;
            }

            if (currentToken == null) {
                currentToken = new SegmentReference(segment, (short) textOffset, (short) displayText.length(), lastX, wordWidth, ascent);
            } else {
                currentToken.updateLimit(currentToken.limit() + displayText.length(), currentToken.width() + wordWidth);
                if (!isEndOfWord && wordStartSegment >= currentLine.segments.size()) {
                    // We added some text to current token, so wordStart should be in this segment, not next
                    //  (`wordStartSegment >= currentLine.segments.size()` - this indicates that a word will start in the next segment, but it should start in the current)
                    wordStartSegment -= 1;
                }
            }

            increaseX(wordWidth);
            textOffset += w.length();

            if (restOfWord != null) {
                // This word was split, replace current word with rest of the word, and run this step again
                words[i] = restOfWord;
                isEndOfWord = true;
                wordStartSegment = 0;
                i -= 1;
                continue;
            }

            if (endsWithNewLine) {
                currentLine.segments.add(currentToken);
                currentToken = null;
                commitLine(lineHeightAscent, lineHeightDescent);
            }
        }
        if (currentToken != null) {
            currentLine.segments.add(currentToken);
        }
    }

    private NodeMeasurer.TextMetrics measure(String text, SegmentStyle style) {
        return nodeMeasurer.measureText(text, style.font);
    }

    public List<LayoutLine> getLines() {
        if (currentLine != null && !currentLine.segments.isEmpty()) {
            commitLine(lineHeightAscent, lineHeightDescent);
        }
        return new ArrayList<>(lines);
    }

    private void commitLine(double commitAscent, double commitDescent) {
        lastX = startX;
        lastY += lineHeightAscent + lineHeightDescent;
        currentLine.commitHeight(commitAscent, commitDescent);
        if (!currentLine.segments.isEmpty()) {
            lines.add(currentLine);
        }
        lineHeightAscent = 0;
        lineHeightDescent = 0;
        currentLine = new LayoutLine(lastY, lineHeightAscent, lineHeightDescent);
        isEndOfWord = true;
        wordStartSegment = 0;
        beforeWordStartDescent = 0;
        beforeWordStartAscent = 0;
        maxDiscoveredX = Math.max(maxDiscoveredX, lastX);
    }

    private void jumpToX(double newXValue) {
        lastX = newXValue;
    }

    private void increaseX(double deltaX) {
        lastX += deltaX;
        maxDiscoveredX = Math.max(maxDiscoveredX, lastX);
    }

    private static int lastWhitespaceIndex(String s) {
        if (s == null || s.isEmpty()) return -1;
        for (int i = s.length() - 1; i >= 0; i--) {
            if (Character.isWhitespace(s.charAt(i))) return i;
        }
        return -1;
    }
}
