package com.tchudyk.text_canvas.renderer;

import com.tchudyk.text_canvas.FxNodes;
import com.tchudyk.text_canvas.layout.LayoutLine;
import com.tchudyk.text_canvas.layout.SegmentReference;
import com.tchudyk.text_canvas.segment.blocks.BaseBlockSegment;
import com.tchudyk.text_canvas.segment.node.NodeSegment;
import com.tchudyk.text_canvas.segment.text.TextSegment;
import javafx.geometry.Bounds;
import javafx.scene.canvas.GraphicsContext;
import javafx.scene.layout.Region;

import java.util.List;

public final class FxRenderer {
    private final GraphicsContext gc;
    private final FxNodes nodesRegistry;

    public FxRenderer(GraphicsContext gc, FxNodes nodesRegistry) {
        this.gc = gc;
        this.nodesRegistry = nodesRegistry;
    }

    public void render(List<LayoutLine> lines, Bounds visibleBounds) {
        for (LayoutLine line : lines) {
            if (line.y > visibleBounds.getMaxY()) {
                unregisterLine(line);
                continue; // Line is after visible bounds, so skip it
            } else if (line.y + line.ascent + line.descent < visibleBounds.getMinY()) {
                unregisterLine(line);
                continue; // Line is before visible bounds, so skip it
            }

            for (SegmentReference segment : line.segments) {
                drawSegment(line, segment, visibleBounds);
            }
        }
    }

    private void unregisterLine(LayoutLine line) {
        for (int i = 0; i < line.segments.size(); i++) {
            SegmentReference segment = line.segments.get(i);
            if (segment.segment() instanceof NodeSegment nodeSegment) {
                nodesRegistry.unregisterFromScreen(nodeSegment.node);
            } else if (segment.segment() instanceof BaseBlockSegment blockSegment) {
                for (LayoutLine innerLine : segment.innerLines()) {
                    unregisterLine(innerLine);
                }
            }
        }
    }

    private void drawSegment(LayoutLine line, SegmentReference ref, Bounds visibleBounds) {
        if (ref.segment() instanceof TextSegment textSegment) {
            String segmentText = textSegment.text();
            String textToDraw = segmentText.substring(ref.offset(), ref.offset() + ref.limit());

            gc.setFont(textSegment.style().font);
            gc.setFill(textSegment.style().fill);
            gc.fillText(textToDraw, ref.x() - visibleBounds.getMinX(), (line.y + line.ascent) - visibleBounds.getMinY());

        } else if (ref.segment() instanceof BaseBlockSegment blockSegment) {
            render(ref.innerLines(), visibleBounds);

            // Draw block border
            double locationX = ref.x() - visibleBounds.getMinX();
            double locationY = line.y + (line.ascent - ref.ascentHeight()) - visibleBounds.getMinY();
            double width = ref.width();
            double height = ref.ascentHeight() + line.descent;
            gc.strokeRect(locationX, locationY, width, height);

        } else if (ref.segment() instanceof NodeSegment segment) {

            nodesRegistry.registerOnScreen(segment.node);
            Region node = segment.node;
            double locationX = ref.x() - visibleBounds.getMinX();
            double locationY = line.y - visibleBounds.getMinY();

            node.relocate(locationX, locationY);
        }
    }
}
